/*
 * Microchip Touchscreen Controller Utility
 *
 * Copyright (c) 2011 Microchip Technology, Inc.
 * 
 * http://www.microchip.com
 */

/*
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/dir.h>

#include <linux/input.h>

#define _POSIX_SOURCE 1 

// Function error return values
#define ERROR_TIMEOUT -1
#define INVALID_BYTE -2

// The tracking ID is an event the PCAP kernel driver returns to indicate the touched point ID
#define TRACKING_ID 57

// Protocol indexes 
#define MCHPTSHARC 0x00
#define MCHPPCAP 0x01
#define MCHPAR1XXX 0x02
#define EVDEV 0x03

// Maximum packet lengths
#define TSHARC_MAX_LENGTH 4
#define MCHIP_MAX_LENGTH 5

// Necessary extern to be able to use scandir function 
extern int alphasort();

// Coordinate and UART state data shared between functions
typedef struct 
{
	unsigned int x;
	unsigned int y;
	unsigned int button;
	unsigned int touch;
	unsigned int trackingID;
	int fd;
	struct termios oldtio,newtio;
} sharedDataType;

// Used to display text string of current protocol
char* getProtocolString(int protocol)
{
	switch (protocol)
	{
		case MCHPTSHARC:
			return "TSHARC";
			break;
		case MCHPPCAP:
			return "PCAP";
			break;
		case MCHPAR1XXX:
			return "AR1XXX";
			break;
		case EVDEV:
			return "EVDEV";
			break;
		default:
			return "<unknown>";
	}
	return NULL;
}

// Used to filter out USB entries within the "/dev/input/by-id" directory
int file_select(const struct direct *entry)
{
    if ((NULL!=strstr(entry->d_name,"Hampshire")) && (NULL!=strstr(entry->d_name,"event")))
    {
	return 1;
    }
    if ((NULL!=strstr(entry->d_name,"Microchip")) && (NULL!=strstr(entry->d_name,"event")))
    {
	return 1;
    }
    else
    {
	return 0;
    }
}

// Used to search the "/dev/input/by-id" directory to determine if a USB controller is attached
int findUSBControllers()
{
    int count;
    int i;
    
    struct direct **files;

    count=scandir("/dev/input/by-id",&files,file_select,alphasort);

    if (0==count)
    {
	return -1;
    }
    
    for (i=1;i<count+1;i++)
	printf("\nFound controller at device path: /dev/input/by-id/%s\n",files[i-1]->d_name);

    return count;
}

// Open device path and set baudrate, port setting
int setPortSettings(sharedDataType* sharedData, char*devicePath, int baudrate)
{
	struct termios oldtio,newtio;

	// Open serial device for reading and writing
	sharedData->fd = open(devicePath, O_RDWR | O_SYNC | O_RSYNC | O_DSYNC);
	if (sharedData->fd < 0) 
	{
//		Commented out since we do not want this displayed while searching for a controller
//		fprintf(stderr,"Error opening port at %s\n",devicePath);
		return -1;
	}
	
	// allow only these baudrates 
	switch (baudrate)
	{
		case 9600:
			baudrate=B9600;
			break;
		case 57600:
			baudrate=B57600;
			break;
		case 115200:
			baudrate=B115200;
			break;
		default:
			fprintf(stderr, "Unsupported baudrate, setting to 9600 baud\n");
			baudrate=B9600;
			break;
	}
	
	// Save current serial port settings
	tcgetattr(sharedData->fd,&oldtio); 

	// Initialize structure for new port settings
	bzero(&newtio, sizeof(newtio)); 
	
	// Setup port flags to set items such as baudrate and flow control
	newtio.c_cflag = baudrate | CS8 | CLOCAL | CREAD;
	
	// Setup output as raw
	newtio.c_oflag = 1;

	// Initialize various other settings	
	newtio.c_cc[VINTR] = 0; // Ctrl-c 
	newtio.c_cc[VQUIT] = 0; // Ctrl-\ 
	newtio.c_cc[VERASE] = 0; // del 
	newtio.c_cc[VKILL] = 0; // @ 
	newtio.c_cc[VEOF] = 4; // Ctrl-d 
	newtio.c_cc[VTIME] = 0; // inter-character timer unused
	newtio.c_cc[VMIN] = 1; // blocking read until 1 character arrives 
	newtio.c_cc[VSWTC] = 0; // '\0' 
	newtio.c_cc[VSTART] = 0; // Ctrl-q 
	newtio.c_cc[VSTOP] = 0; // Ctrl-s 
	newtio.c_cc[VSUSP] = 0; // Ctrl-z 
	newtio.c_cc[VEOL] = 0; // '\0' 
	newtio.c_cc[VREPRINT] = 0; // Ctrl-r 
	newtio.c_cc[VDISCARD] = 0; // Ctrl-u 
	newtio.c_cc[VWERASE] = 0; // Ctrl-w 
	newtio.c_cc[VLNEXT] = 0; // Ctrl-v 
	newtio.c_cc[VEOL2] = 0; // '\0' 
	
	//Flush current setttings and activate new settings for port
	tcflush(sharedData->fd, TCIFLUSH);
	tcsetattr(sharedData->fd,TCSANOW,&newtio);

	return 0;
}

// Restore port setting to original state
int restorePortSettings(sharedDataType* sharedData)
{
	// restore the old port settings 
	tcsetattr(sharedData->fd,TCSANOW,&sharedData->oldtio);

	close(sharedData->fd);

	return 0;
}

// Set the UART hardware timeout
int setTimeout(sharedDataType* sharedData, unsigned int milliSeconds)
{
	struct termios serialSettings;

	// get current port settings
	tcgetattr(sharedData->fd,&serialSettings);
	serialSettings.c_cc[VMIN]=0;
	// convert to deciseconds
	serialSettings.c_cc[VTIME]=milliSeconds / 100;
	tcsetattr(sharedData->fd,TCSANOW,&serialSettings);
	return 0;
}

// Decode packets of data from a device path using EVDEV protocol
int decodeEVDEVPacket(sharedDataType* sharedData, struct input_event *evdevPacket)
{
	int returnValue=1;
		
	// Since we have copied the evdev packet's information into the
	// input_event structure, we may use and apply it's properties directly.
	if (EV_ABS==evdevPacket->type)
	{
		switch (evdevPacket->code)
		{
			case ABS_X:
				sharedData->x = evdevPacket->value;
			break;

			case ABS_Y:
				sharedData->y = evdevPacket->value;
			break;
		}
	}

	else if (EV_KEY==evdevPacket->type)
	{
		sharedData->button = evdevPacket->value;
	}
	else
	{	
		returnValue=0;			
	}

	return returnValue;
}

// Decode packets of data from a device path using AR1XXX protocol
// Returns 1 if a full packet is available, zero otherwise
int decodeAR1XXXPacket(sharedDataType* sharedData, char* packet, int packetSize, int *index, char data)
{
	int returnValue=0;

	if (packetSize < MCHIP_MAX_LENGTH)
	{
		fprintf(stderr, "Packet size too small.\n");
		exit(1);
	}
	packet[*index] = data;

	/****************************************************
	
	Data format, 5 bytes: SYNC, DATA1, DATA2, DATA3, DATA4
	
	SYNC [7:0]: 1,0,0,0,0,TOUCHSTATUS[0:0]
	DATA1[7:0]: 0,X-LOW[6:0]
	DATA2[7:0]: 0,X-HIGH[4:0]
	DATA3[7:0]: 0,Y-LOW[6:0]
	DATA4[7:0]: 0,Y-HIGH[4:0]
	
	TOUCHSTATUS: 0 = Touch up, 1 = Touch down
	
	****************************************************/		
	
	switch ((*index)++) {
		case 0:
			if (!(0x80 & data))
			{
			    // Sync bit not set
			    *index = 0;			  
			    returnValue=INVALID_BYTE;
			}
			break;

		case (MCHIP_MAX_LENGTH - 1):
			// verify byte is valid for current index 
			if (0x80 & data)
			{
				// byte not valid
				packet[0]=data;
				*index = 1;
				break;
			}			  

			*index = 0;

			sharedData->x = ((packet[0] & 0x38) << 6) | ((packet[1] & 0x7f) << 2) | ((packet[3] & 0x0c) >> 2);
			sharedData->y = ((packet[0] & 0x07) << 9) | ((packet[2] & 0x7f) << 2) | ((packet[3] & 0x03));
			sharedData->button = (packet[0] & 0x40) >> 6;			

			sharedData->x = ((packet[2] & 0x1f) << 7) | (packet[1] & 0x7f);
			sharedData->y, ((packet[4] & 0x1f) << 7) | (packet[3] & 0x7f);
			sharedData->button = 0 != (packet[0] & 1);

			returnValue=1;
			break;
		default:
			// verify byte is valid for current index
			if (0x80 & data)
			{
				// byte not valid
				packet[0]=data;
				*index = 1;
			}			  
			break;
			
	}

	return returnValue;
}

// Decode packets of data from a device path using PCAP protocol
// Returns 1 if a full packet is available, zero otherwise
int decodePCAPPacket(sharedDataType* sharedData, char* packet, int packetSize, int *index, char data)
{
	int returnValue=0;

	if (packetSize < MCHIP_MAX_LENGTH)
	{
		fprintf(stderr, "Packet size too small.\n");
		exit(1);
	}
	packet[*index] = data;

	/****************************************************
	
	Data format, 5 bytes: SYNC, DATA1, DATA2, DATA3, DATA4
	
	SYNC [7:0]: 1,TOUCHID[6:5],0,0,TOUCHSTATUS[2:0]
	DATA1[7:0]: 0,X-LOW[6:0]
	DATA2[7:0]: 0,X-HIGH[2:0]
	DATA3[7:0]: 0,Y-LOW[6:0]
	DATA4[7:0]: 0,Y-HIGH[2:0]
	
	TOUCHSTATUS: 0 = Touch up, 1 = Touch down
	
	****************************************************/		
	
	switch ((*index)++) {
		case 0:
			if (!(0x80 & data))
			{
			    // Sync bit not set
			    *index = 0;			  
			    returnValue=INVALID_BYTE;
			}
			break;

		case (MCHIP_MAX_LENGTH - 1):		  			
			// verify byte is valid for current index 
			if (0x80 & data)
			{
				// byte not valid 			  
				packet[0]=data;
				*index = 1;
				returnValue=INVALID_BYTE;
				break;
			}

			*index = 0;
			
			sharedData->trackingID =(packet[0]&0x60) >> 5;
			sharedData->x = ((packet[2] & 0x1f) << 7) | (packet[1] & 0x7f);
			sharedData->y = ((packet[4] & 0x1f) << 7) | (packet[3] & 0x7f);
			sharedData->button = (0 != (packet[0] & 7));

			returnValue=1;
			break;
		default:
			// verify byte is valid for current index 
			if (0x80 & data)
			{
				// byte not valid
				packet[0]=data;
				*index = 1;
				returnValue=INVALID_BYTE;
			}		    
			break;
	}
	return returnValue;
}

// Decode packets of data from a device path using TSHARC protocol
// Returns 1 if a full packet is available, zero otherwise
int decodeTSHARCPacket(sharedDataType* sharedData, char* packet, int packetSize, int *index, char data)
{
	int returnValue=0;

	if (packetSize < TSHARC_MAX_LENGTH)
	{
		fprintf(stderr, "Packet size too small.\n");
		exit(1);
	}
	packet[*index] = data;

	/****************************************************
	
	Data format, 4 bytes: SYNC, DATA1, DATA2, DATA3
	
	SYNC [7:0]: 1,TOUCH,X[11:9],Y[11:9]
	DATA1[7:0]: 0,X[8:2]
	DATA2[7:0]: 0,Y[8:2]
	DATA3[7:0]: 0,0,0,0,X[1:0],Y[1:0]
	
	TOUCH: 0 = Touch up, 1 = Touch down
	
	****************************************************/	

	switch ((*index)++) {
		case 0:
			if (!(0x80 & data))
			{
			    *index = 0;		
    			    returnValue=INVALID_BYTE;
	  
			}
			break;

		case (TSHARC_MAX_LENGTH - 1):
			*index = 0;
	
			// verify byte is valid for current index
			if (0x80 & data)
			{
				// byte not valid
				packet[0]=data;
				*index = 1;
				returnValue=INVALID_BYTE;
				break;
			}
	
			sharedData->x = ((packet[0] & 0x38) << 6) | ((packet[1] & 0x7f) << 2) | ((packet[3] & 0x0c) >> 2);
			sharedData->y = ((packet[0] & 0x07) << 9) | ((packet[2] & 0x7f) << 2) | ((packet[3] & 0x03));
			sharedData->button = (packet[0] & 0x40) >> 6;

			returnValue=1;

			break;
		default:
			// verify byte is valid for current index 
			if (0x80 & data)
			{
				// byte not valid 			  
				packet[0]=data;
				*index = 1;
				returnValue=INVALID_BYTE;
			}

			break;
	}

	return returnValue;
}

// Decode packets of data from a device path using specified protocol or optionally search for device path using specified protocol
int decode(sharedDataType* sharedData, char *devicePath, int protocol, int baudrate, int findmode, int maxPacketsToRead)
{
	size_t num_read_bytes=0;
	int byte=0;
	char packet[5];
	int index=0;
	int ret=0;
	char name[200];
	struct input_event evdevPacket[100]; // read up to 100 evdev packets
	int timeoutInMilliseconds=5000;
	int numPacketsRecieved=0;
	int numBadBytes=0;
	int maxBadBytes=5;

	sharedData->x=0;
	sharedData->y=0;
	sharedData->button=0;

	// If we are looking for controller, we want to decrease this value to a value just big enough 
	// for a good verification of packet data
	if (1==findmode)
	{
		timeoutInMilliseconds=500;
	}

	// Set port settings for all paths except "/dev/input/by-id" which is not a settable location
	if (0!=strcmp(devicePath,"/dev/input/by-id"))
	{
		if (setPortSettings(sharedData,devicePath,baudrate) < 0)
		{
			return -1;
		};
	}

	// we handle EVDEV searchs by searching "/dev/input/by-id" files for "Hampshire" or "Microchip" keywords
	if ((EVDEV==protocol) && (1==findmode))
	{
		// if we are processing event IDs in "/dev/input" do this first section,
		// otherwise, do the second section.
		if (0!=strcmp(devicePath,"/dev/input/by-id"))
		{
			ioctl(sharedData->fd, EVIOCGNAME(sizeof(name)), name);
	//		Uncomment the next line to get a full list of kernel input names
	//		printf("Input device name: \"%s\"\n", name);
			if ((strstr(name,"Hampshire")) || (strstr(name,"Microchip")) || (strstr(name,"AR1020")))
			{
				return 1;
			}
			else
			{
				return -1;
			}
		}
		else
		{
			// display USB controllers with a path in /dev/input/by-id
			return findUSBControllers();
		}
		
	}

	// Setting timeouts does not work on paths using EVDEV protocol
	else if (EVDEV!=protocol)
	{	
		setTimeout(sharedData, timeoutInMilliseconds);
	}

	// Output decoded packets if we are testing for valid coordinates with specified device path
	if (0==findmode)
	{
		printf("\nDecoding %s packets (Press Ctrl-C to exit)\n",getProtocolString(protocol));
	}

	while (1) 
	{ 
		if (EVDEV==protocol)
		{
			num_read_bytes = read(sharedData->fd, evdevPacket, sizeof(struct input_event) * 100);

			if (num_read_bytes < (int) sizeof(struct input_event)) {
			    fprintf(stderr,"Did not read enough bytes to populate evdev packet.  Skipping this packet.\n");
			    continue;
			}
		}
		else
		{
			num_read_bytes = read(sharedData->fd,&byte,1);
//			Uncomment the section below to monitor data byte stream
//			if (0!=num_read_bytes)
//			{
//				printf("byte: 0x%02x\n",byte);
//			}
		}

		// if no bytes read (which is expected if there is no touch), then determine how we handle this
		// depending on whether we are searching for a controller or not.
		if (0==num_read_bytes)
		{
			// If we are not searching for a valid device path, allow the function to continue looping.
			if (timeoutInMilliseconds >= 5000)
			{
			 	continue;
			}
			else
			{
				// A byte read timeout has occured
				index = 0;
				return ERROR_TIMEOUT;
			}
		}
		else
		{
			// retrieve packet from device path based on current protocol
			switch (protocol)
			{

				case MCHPTSHARC:
					ret=decodeTSHARCPacket(sharedData, packet, sizeof(packet), &index, byte);
					break;
				case MCHPPCAP:
					ret=decodePCAPPacket(sharedData, packet, sizeof(packet), &index, byte);
					break;
				case MCHPAR1XXX:
					ret=decodeAR1XXXPacket(sharedData, packet, sizeof(packet), &index, byte);
					break;
				case EVDEV:
					{
						int num_packets=(int) (num_read_bytes / sizeof(struct input_event));
						int i;
						for (i = 0; i < num_packets; i++)
						{
							ret=decodeEVDEVPacket(sharedData, &evdevPacket[i]);
							// To ensure every packet is reported, the evdev packet
							// coordinates will be display during processing
							// evdev packet array.
							if ((ret > 0) && (0==findmode) && (0!=sharedData->x) && (0!=sharedData->y))
							{
								if (MCHPPCAP != protocol)
								{
									printf("State: %u %u %d\n",sharedData->x,sharedData->y,sharedData->button);
								}
								else
								{
									printf("State: ID %u - %u %u %d\n",sharedData->trackingID,sharedData->x,sharedData->y,sharedData->button);
								}
							}
						}
					}
					break;
				default:
					fprintf(stderr,"Invalid protocol id set.\n");
					exit(1);
					break;
			}

			// ret is positive if we have decoded a packet. If we are testing a device path, output coordinate here.
			if (ret > 0)
			{
				if ((0==findmode) && (EVDEV != protocol))
				{
					if (MCHPPCAP != protocol)
					{
						printf("State: %u %u %d\n",sharedData->x,sharedData->y,sharedData->button);
					}
					else
					{
						printf("State: ID %u - %u %u %d\n",sharedData->trackingID,sharedData->x,sharedData->y,sharedData->button);
					}
				}
				numPacketsRecieved+=ret;
			}

			// Return if there was a communication error.  This is useful for testing device paths.
			else if (ret < 0)
			{
				if (INVALID_BYTE == ret)
				{			
					if (0==findmode)
					{	
						printf("Invalid byte detected (value 0x%02x at index %d)\n",byte,index);
					}
					numBadBytes++;
					if (numBadBytes > maxBadBytes)
					{
						return -1;
					}
				}
				else
				{
					return ret;
				}
			}
			// processing packet byte
			else
			{				
//				printf("Good byte 0x%02x at index %d\n",byte,index);
			}

			// if maxPacketsToRead set, decode only up to maxPacketsToRead value
			if ((0 != maxPacketsToRead) && (numPacketsRecieved >= maxPacketsToRead))
			{
				// Max packet read threshold hit.  Return number of packets correctly decoded.
				return numPacketsRecieved;
			}


		}

	}

	// restore the old port settings followed by closing port
	restorePortSettings(sharedData);

   return 0;
}

// If invalid or no command parameters set, display this help text.
void printHelpText(char* appName)
{
	printf("\nUsage: %s -d <device path> [-p] <protocol> [-f] [-h]\n\n", appName);
	printf("-d <device path> - (Required) System device path of controller.  For example: /dev/ttyS0 is a device path.\n");
	printf("-p <protocol> - (Required) System device path of controller.  Valid values are TSHARC, PCAP, AR1XXX, EVDEV.\n");
	printf("-b <baudrate> - (Optional) Sets baudrate of communication.  Valid values are 9600, 57600, 115200.  Default value is 9600 baud. \n");
	printf("-f - (Optional) Search common device paths for protocol response.  The touchscreen need to be\n");
	printf("in a touched state in order for this option to return a device path.\n");
	printf("-h - (Optional) Shows this help text.\n\n");
}

int main(int argc, char **argv)
{
	int i;
	int c;
	sharedDataType sharedData;
	int protocol=-1;
	int baudrate=9600;
	int numPacketsRecieved;
	int findmode=0;
	char devicePath[100]="";
		
	printf("Microchip Touchscreen Controller Utility V1.00\n\n");

	if (1!=argc)
	{
		printf("Command-line options selected:\n");
	}
	
	// process command line parameters
	while ((c = getopt (argc, argv, "d:p:b:fh")) != -1)     
	{
		switch (c)
		{
			// device path parameter
			case 'd':
				printf("Device path set to %s\n",optarg);
				strcpy(devicePath,optarg);
				break;

			// protocol parameter
			case 'p':
				printf("Protocol set to %s\n",optarg);
				if (0==strcmp(optarg,"TSHARC"))
				{
					protocol=MCHPTSHARC;
				}
				else if (0==strcmp(optarg,"PCAP"))
				{
					protocol=MCHPPCAP;
				}
				else if (0==strcmp(optarg,"AR1XXX"))
				{
					protocol=MCHPAR1XXX;
				}
				else if (0==strcmp(optarg,"EVDEV"))
				{
					protocol=EVDEV;
				}
				else
				{
					fprintf(stderr, "Invalid protocol %s specified\n",optarg);
					printHelpText(argv[0]);
				}
				sscanf(optarg,"%d",&protocol);
				break;

			// baudrate parameter
			case 'b':
				printf("Setting baudrate to %s\n",optarg);
				sscanf(optarg,"%d",&baudrate);
				break;  


			// flag mode paramter
			case 'f':
				// look for controller on common device paths for controller 			
				// response that matches protocol
				printf("Find mode enabled.\n");
				findmode=1;

				break;  

			default:
			break;
		}
    }

    for (i = optind; i < argc; i++)
     printf ("Non-option argument %s\n", argv[i]);

	// No parameters set, so display help
	if (argc == 1) {
		printHelpText(argv[0]);
		exit(1);
	}

	// If set, we will search all common device paths for a touch packet response 
	if (1==findmode)
	{
		int ret;
		int maxSerialID=100;

		if (EVDEV != protocol)
		{
			for (i=0;i<maxSerialID;i++)
			{
				snprintf(devicePath,sizeof(devicePath),"/dev/ttyS%d",i);
				ret=decode(&sharedData, devicePath, protocol, baudrate, findmode, 5);
				if (ret >0)
				{
					printf("\nFound controller at device path: %s\n\n",devicePath);
					exit(0);
				}
				else
				{
	// 				Uncomment to display paths the controller was not detected at
	//				printf("not found at %s\n",devicePath);
				}
			}

			if (maxSerialID==i) 
			{
				printf("\nNo controllers found.\n\n");
				printf("Note: The touchscreen needs to be touched in find mode in order for the controller's device path to be discovered.\n\n");
			}
		}
		else
		{
			int foundController=0;

			for (i=0;i<maxSerialID;i++)
			{
				snprintf(devicePath,sizeof(devicePath),"/dev/input/event%X",i);
				ret=decode(&sharedData, devicePath, protocol, baudrate, findmode, 0);
				if (ret >0)
				{
					printf("\nFound controller at device path: %s\n",devicePath);
					foundController=1;
				}
				else
				{
	//				printf("not found at %s\n",devicePath);
				}
			}

			snprintf(devicePath,sizeof(devicePath),"/dev/input/by-id",i);
			ret=decode(&sharedData, devicePath, protocol, baudrate, findmode, 0);

			// If we have decoded one or more packets at device path, we have discovered the controller
			if (ret >0)
			{
				foundController=1;
			}

			// Display message if no controllers found
			if (0==foundController) 
			{
				printf("\nNo controllers found.\n\n");
				printf("Please verify that the controller is either attached (for USB) or activated (for UART). Touching the touchscreen in find mode is not a requirement for automatically discovering the controller's evdev device path.\n\n");
			}
			else
			{
				printf("\n");
			}
		}
	}
	else
	{
		// Require that device path is set if we are not in find mode (need to know where to test)
		if (0==strcmp(devicePath,""))
		{
			fprintf(stderr,"Device path not set\n");
			printHelpText(argv[0]);
			exit(1);
		}

		// Require that protocol is set if we are not in find mode (need to know decoding to use)
		if (-1==protocol)
		{
			fprintf(stderr,"Controller protocol not set\n");
			printHelpText(argv[0]);
			exit(1);
		}

		// Test packets at device path or generate a device path using specified protocol
		decode(&sharedData, devicePath, protocol, baudrate, findmode, 0);
	}

	return 0;
}

