/*****************************************************************************
* CODE OWNERSHIP AND DISCLAIMER OF LIABILITY
* 
* Microchip Technology Incorporated ("Microchip") retains all ownership and 
* intellectual property rights in the code accompanying this message and in 
* all derivatives hereto.  You may use this code, and any derivatives created 
* by any person or entity by or on your behalf, exclusively with Microchip's 
* proprietary products.  Your acceptance and/or use of this code constitutes 
* agreement to the terms and conditions of this notice.
* 
* CODE ACCOMPANYING THIS MESSAGE IS SUPPLIED BY MICROCHIP "AS IS".  NO 
* WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT 
* LIMITED TO, IMPLIED WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND 
* FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS CODE, ITS INTERACTION WITH 
* MICROCHIP'S PRODUCTS, COMBINATION WITH ANY OTHER PRODUCTS, OR USE IN ANY 
* APPLICATION. 
* 
* YOU ACKNOWLEDGE AND AGREE THAT, IN NO EVENT, SHALL MICROCHIP BE LIABLE, 
* WHETHER IN CONTRACT, WARRANTY, TORT (INCLUDING NEGLIGENCE OR BREACH OF 
* STATUTORY DUTY), STRICT LIABILITY, INDEMNITY, CONTRIBUTION, OR OTHERWISE, 
* FOR ANY INDIRECT, SPECIAL, PUNITIVE, EXEMPLARY, INCIDENTAL OR CONSEQUENTIAL 
* LOSS, DAMAGE, FOR COST OR EXPENSE OF ANY KIND WHATSOEVER RELATED TO THE 
* CODE, HOWSOEVER CAUSED, EVEN IF MICROCHIP HAS BEEN ADVISED OF THE 
* POSSIBILITY OR THE DAMAGES ARE FORESEEABLE.  TO THE FULLEST EXTENT ALLOWABLE 
* BY LAW, MICROCHIP'S TOTAL LIABILITY ON ALL CLAIMS IN ANY WAY RELATED TO THIS 
* CODE, SHALL NOT EXCEED THE PRICE YOU PAID DIRECTLY TO MICROCHIP SPECIFICALLY 
* TO HAVE THIS CODE DEVELOPED.
* 
* You agree that you are solely responsible for testing the code and 
* determining its suitability.  Microchip has no obligation to modify, test, 
* certify, or support the code.
*
* Author                Date        Comment
*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
* Steve Grahovac		04/05/10	Initial Microchip Touch CE example code
*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
* 
* Description:
*    This source file implements the UART specific code to enable communication
*    with the touch device.  
* 
******************************************************************************/

#include <windows.h>
#include <types.h>
#include <nkintr.h>
#include <tchddsi.h>
#include "MCUART.h"

// Used by ReceiveResponseUART() function to filter incoming bytes until the appropriate
// start byte has been matched
#define COMMAND_RESPONSE 1
#define PACKET_RESPONSE 2

// UART port used - platform specific
// Please modify the following line to best suit target platform.
#define UART_PORT L"COM2:"

/** Variables **/

// The following extern variables all originate in the WinCE platform builder file "tchmain.c".

// Pointer to the GWES callback function that is responsible for moving the cursor.
extern "C" PFN_TOUCH_PANEL_CALLBACK v_pfnPointCallback;
// The handle used in tchmain.c to wait for valid calibration point.
extern "C" HANDLE  hCalibrationSampleAvailable;
// The data packet coordinate used to store current calibration target in tchmain.c.
extern "C" LONG    lCalibrationXCoord; 
extern "C" LONG    lCalibrationYCoord; 
// Needed to manage proper state using TouchPanelReadCalibrationPoint() function of "tchmain.c" and our thread
extern "C" INT     CalibrationState;
//  Used to signal Power Manager activity event if this been set up during calibration
extern "C" HANDLE  ghevCalibrationActivity;  

static TOUCH_DEVICE gTouchDevice =
{
	0,		// UART handle
	0,		// UART event monitor thread handle
	0,		// Event handle to signal that it is safe to monitor for packets
	FALSE	// Boolean to flag whether data is available or not
};

/** Prototypes **/
void SetCalibrationPoint(LONG currentCalibrationXCoord, LONG currentCalibrationYCoord);
static ULONG WaitForUARTData(PVOID data);

/******************************************************************************
	Function:
		SetCalibrationState()

	Description:
		Sets the current calibration state
******************************************************************************/
void SetCalibrationState(INT currentCalibrationState)
{
	CalibrationState=currentCalibrationState;
}

/******************************************************************************
	Function:
		SetCalibrationPoint()

	Description:
		Sets the coordinates for the current calibration point
******************************************************************************/
void SetCalibrationPoint(LONG currentCalibrationXCoord, LONG currentCalibrationYCoord)
{
	lCalibrationXCoord=currentCalibrationXCoord; 
	lCalibrationYCoord=currentCalibrationYCoord; 
}

/******************************************************************************
	Function:
		WaitForUARTData()

	Description:
		This is the packet monitoring thread.  This threads waits for a UART
		event and then reads data / processes packets until no other data 
		is available.
******************************************************************************/
static ULONG WaitForUARTData(PVOID data)
{
	DWORD dwCommMask=0;
	DWORD returnValue=0;
	INT UncalX=0;
	INT UncalY=0;
	TOUCH_PANEL_SAMPLE_FLAGS TipStateFlags = TouchSampleValidFlag;
	PFN_TOUCH_PANEL_CALLBACK pfnCallback;
	LONG lastCalibrationXCoord=0;
	LONG lastCalibrationYCoord=0;
	INT32 CalX=0;
	INT32 CalY=0;

	DEBUGMSG(1,(TEXT("WaitForUARTData: begin\r\n")));

	// wait until initialization is complete
	WaitForSingleObject(gTouchDevice.hInitializationComplete,INFINITE);
	CloseHandle(gTouchDevice.hInitializationComplete);
	gTouchDevice.hInitializationComplete=NULL;

	while (1)
	{
		// Cause calibration to wait for a pen-up packet on every calibration target
		ResetEvent(hCalibrationSampleAvailable);
		// this function will wait until a character is received and placed in input buffer
		// before proceeding (wait behavior specified by SetCommMask() function)
		returnValue=WaitCommEvent (gTouchDevice.hUART, &dwCommMask, 0); 

		// The WaitCommEvent will fail after closing the UART handle due to 
		// Destroy() function being called.  We want to exit thread in this scenario.
		if (FALSE==returnValue)
		{
			DEBUGMSG(1,(TEXT("WaitForUARTData: failed waiting for communication event\r\n")));
			break;
		}

		// we have received a character from FIFO, so data is pending
		gTouchDevice.dataPending=TRUE;

		// we want to read/process data until no data is available in UART buffer
		do
		{
			TipStateFlags=TouchSampleValidFlag;

			// We assign a pointer to the callback function that lives in GWES
			// to move the cursor.
			pfnCallback=v_pfnPointCallback;
			if (NULL==v_pfnPointCallback)
			{
				break;
			}

			returnValue=GetPacket(&UncalX, &UncalY, &TipStateFlags);

			//  Used to signal Power Manager activity event if this been set up during calibration
            if (CalibrationState &&( ghevCalibrationActivity != NULL))
            {
               	SetEvent(ghevCalibrationActivity);
            }

			// The GetPacket() function returns FALSE when a pen-up event has occurs
			if (FALSE==returnValue)
			{
				// Only set calibration point if there is a pen-down to pen-up transition
				if (CalibrationState && (CalibrationDown==CalibrationState))
				{
					SetCalibrationPoint(UncalX, UncalY);
				}
			}
			else if (CalibrationState)
			{
				CalibrationState=CalibrationDown;
			}

			// Takes an uncalibrated point and applies calibration parameters
			// If no calibration exists, CalX/CalyY will be the same as UncalX/UncalY
			TouchPanelCalibrateAPoint( UncalX, UncalY, &CalX, &CalY );

			(pfnCallback)( TipStateFlags, CalX, CalY);		

		} while (TRUE==gTouchDevice.dataPending);

		// if we are processing a calibration, send signal to TouchPanelReadCalibrationPoint() of tchmain.c
		// to cause the calibration to process next point
		if (CalibrationState)
		{
			// Set to calibration valid since we have a valid point
			SetCalibrationState(CalibrationValid);
			SetEvent(hCalibrationSampleAvailable);
		}

		DEBUGMSG(1,(TEXT("WaitForUARTData: Packet processing complete\r\n")));
	}

	return 0;
}

/******************************************************************************
	Function:
		WaitForUARTData()

	Description:
		Sets up UART for communication with touch device.
******************************************************************************/
BOOL Init()
{
	UINT8 numReceivedBytes=4;
	UINT32 registerStartAddress=0;

	DCB currentDCB;
	BYTE byte=0xFE;
	int index=0;

	DEBUGMSG(1, (TEXT("Init: begin\r\n")));

	gTouchDevice.hUART=CreateFile( UART_PORT,       // name of com port - platform specific
										GENERIC_READ |GENERIC_WRITE, // access
										0,                          // share 0:cannot share the COM port
										NULL,                       // security NULL:none
										OPEN_EXISTING,              // no creation, must be there
										FILE_FLAG_NO_BUFFERING | FILE_FLAG_WRITE_THROUGH,//0,//FILE_FLAG_NO_BUFFERING,      
										NULL);

	if (INVALID_HANDLE_VALUE==gTouchDevice.hUART) 
	{

		DEBUGMSG(1, (TEXT("Init: Error opening UART port\r\n")));
	}

	GetCommState(gTouchDevice.hUART,&currentDCB);
	currentDCB.Parity=NOPARITY;
	currentDCB.StopBits=ONESTOPBIT;
	currentDCB.BaudRate=CBR_9600;
	if(!SetCommState(gTouchDevice.hUART,&currentDCB))
	{
	   DEBUGMSG(1, (TEXT("Init: SetCommState error %d\r\n"), GetLastError()));
		CloseHandle(gTouchDevice.hUART);
		gTouchDevice.hUART = NULL;
	}

	// Waiting on RX char event only
	if ( !SetCommMask (gTouchDevice.hUART, EV_RXCHAR) ) 
	{
		DEBUGMSG (1, (TEXT("Error from SetCommMask  %d\r\n"), GetLastError()));
        return (FALSE);
	}

	gTouchDevice.hInitializationComplete = CreateEvent( NULL, FALSE, FALSE, NULL);

	if ( !gTouchDevice.hInitializationComplete )
		return ( FALSE );

	if (!(gTouchDevice.hThread = CreateThread( NULL, 0, WaitForUARTData, 0, 0, NULL))) {

		DEBUGMSG (1, (TEXT("Error from CreateThread  %d\r\n"), GetLastError()));
        return (FALSE);
	} 
	else
	{
		// Set our interrupt thread's priority
		CeSetThreadPriority(gTouchDevice.hThread, 109);
	}

	UINT8 sendTouchOptionBytes[]={0x55,0x5,0x21,0x00,0x0d,0x01,0x00};
	UINT8 sendRegisterStartAddressBytes[]={0x55,0x1,0x22};
	UINT8 sendDisableBytes[]={0x55,0x1,0x13};
	UINT8 sendEnableBytes[]={0x55,0x1,0x12};
	UINT8 receiveBytes[100];

	// Suspend our packet monitoring thread
	SuspendThread(gTouchDevice.hThread);

	// Flush out any existing data in the UART receive buffer
	numReceivedBytes=100;
	ReceiveResponseUART(receiveBytes,&numReceivedBytes,500,0);

	// Disable controller touch
	numReceivedBytes=4;
	SendCommandUART(sendDisableBytes,3);
	Sleep(50);
	ReceiveResponseUART(receiveBytes,&numReceivedBytes,250,COMMAND_RESPONSE);


	// Retrieve the register start address
	numReceivedBytes=5;
	SendCommandUART(sendRegisterStartAddressBytes,3);
	ReceiveResponseUART(receiveBytes,&numReceivedBytes,250,COMMAND_RESPONSE);

	// Set ram value to put controller into raw mode.
	registerStartAddress=receiveBytes[4];
	numReceivedBytes=4;
	// Add register start address to modify register at appropriate offset
	sendTouchOptionBytes[4]+=registerStartAddress;
	SendCommandUART(sendTouchOptionBytes,7);
	ReceiveResponseUART(receiveBytes,&numReceivedBytes,250,COMMAND_RESPONSE);

	// Enable controller touch
	numReceivedBytes=4;
	SendCommandUART(sendEnableBytes,3);
	ReceiveResponseUART(receiveBytes,&numReceivedBytes,250,COMMAND_RESPONSE);

	// Resume our packet monitoring thread
	ResumeThread(gTouchDevice.hThread);

	TouchPanelSetCalibration( 0, NULL, NULL, NULL, NULL );

    return TRUE;
}

/******************************************************************************
	Function:
		GetPacket()

	Description:
		Retrieve one packet from touch device.
******************************************************************************/
BOOL GetPacket(
    INT *xpos,
    INT *ypos,
	TOUCH_PANEL_SAMPLE_FLAGS *pTipStateFlags
)
{
	UINT32 control=0;
	UINT8 readBytes[5];
	int i;
	UINT8 numBytesToReceive=5;
	UINT8 numRetries=5;
	BOOL returnValue=FALSE;

	// initialize start byte
	for (i=0;i<5;i++)
	{
		readBytes[i]=0;
	}

	// read bytes until we receive start byte with sync bit set (0x80)
	if (FALSE==ReceiveResponseUART(readBytes, &numBytesToReceive, 100,PACKET_RESPONSE))
	{		
		DEBUGMSG(1,(TEXT("GetPacket: Data read failure on start byte %d 0x%x\r\n"),numBytesToReceive,readBytes[0]));
	}
	else
	{
		returnValue=TRUE;
	} 

	if (TRUE==returnValue)
	{

		// packet has pen down bit set
		if (readBytes[0] & 1)
		{
			*pTipStateFlags=TouchSampleDownFlag | TouchSampleValidFlag;;
		}
		// packet does not have pen down bit set, so it is a pen up
		else
		{
			*pTipStateFlags=TouchSampleValidFlag;
		}

		*xpos= ((readBytes[2] & 0x1f)<<7) | (readBytes[1] & 0x7f);
		*ypos= ((readBytes[4] & 0x1f)<<7) | (readBytes[3] & 0x7f);

		DEBUGMSG(1,(TEXT("%d x %d y %d\r\n"),(1 & readBytes[0])!=0,*xpos,*ypos));
	}
	else
	{
		// If there was a communication error, set pen state to up to avoid a potential 
		// stuck pen-down state
		*pTipStateFlags=TouchSampleValidFlag;
		// UART buffer is clear
		gTouchDevice.dataPending=FALSE;
	}

	return returnValue;
}

/******************************************************************************
	Function:
		SendCommandUART()

	Description:
		Send bytes to touch device using UART interface.
******************************************************************************/
BOOL SendCommandUART(UINT8* sendBytes,UINT8 numBytesToSend)
{
	DWORD dwNumBytesWritten;
	BOOL returnValue=FALSE;
	COMMTIMEOUTS commTimeout;

	DEBUGMSG(1,(TEXT("SendCommandUART: begin\r\n")));

	GetCommTimeouts(gTouchDevice.hUART, &commTimeout);

	commTimeout.WriteTotalTimeoutConstant=2;
	commTimeout.WriteTotalTimeoutMultiplier=0;

	SetCommTimeouts(gTouchDevice.hUART, &commTimeout);

	returnValue= WriteFile(gTouchDevice.hUART,(void *)(sendBytes),  
				numBytesToSend, // how many to write
				&dwNumBytesWritten,
				NULL);

	DEBUGMSG(1,(TEXT("SendCommandUART: %d 0x%x 0x%x 0x%x\r\n"),dwNumBytesWritten,sendBytes[0],sendBytes[1],sendBytes[2]));
	return returnValue;
}

/******************************************************************************
	Function:
		ReceiveResponseUART()

	Description:
		Receive bytes from touch device using UART interface.
******************************************************************************/
BOOL ReceiveResponseUART(UINT8* responseBytes, UINT8* numBytesToReceive, DWORD timeoutInMilliseconds, DWORD Flags)
{
	COMMTIMEOUTS commTimeout;
	BOOL returnValue;
	UINT8 i;
	UINT8 j;
	DWORD receiveCount=0;

	DEBUGMSG(1,(TEXT("ReceiveResponseUART: begin\r\n")));

	GetCommTimeouts(gTouchDevice.hUART, &commTimeout);

	commTimeout.ReadIntervalTimeout = 0;
	commTimeout.ReadTotalTimeoutConstant = timeoutInMilliseconds;
    commTimeout.ReadTotalTimeoutMultiplier = 1;

	SetCommTimeouts(gTouchDevice.hUART, &commTimeout);

	for (i=0;i<*numBytesToReceive;i++)
	{
		responseBytes[i]=0;
	}

	for (i=0;i<*numBytesToReceive;i++)
	{
		// ignore leading values other than 0x55 for command response
		if ((0x55 != responseBytes[0]) && (1==i) && (COMMAND_RESPONSE & Flags))
		{
			i--;
		}
		// ignore leading values of bytes where sync bit is not set
		else if (!(0x80 & responseBytes[0]) && (1==i) && (PACKET_RESPONSE & Flags))
		{
			i--;
		}

		returnValue= ReadFile(gTouchDevice.hUART,(void *)(&responseBytes[i]),  
					1, // how many to read
					(DWORD*)&receiveCount, //number of bytes read
					NULL);

		if ((0 == receiveCount) || (FALSE==returnValue))
		{
			DEBUGMSG(1,(TEXT("ReceiveResponseUART: Zero bytes received\r\n")));
			// record the actual bytes received
			*numBytesToReceive=i;
			returnValue=FALSE;
			break;
		}
	}

	for (j=0;j<*numBytesToReceive;j++)
	{
		DEBUGMSG(1,(TEXT("%d-%x\r\n"),j,responseBytes[j]));
	}

	return returnValue;
}


/******************************************************************************
	Function:
		Destroy()

	Description:
		Close connection and free memory on current touch interface.
******************************************************************************/
void Destroy()
{
	if (gTouchDevice.hUART != NULL)
	{
		// Since we are not hooking an interrupt, but rather monitoring UART events, 
		// the WinCE architecture will call DdsiTouchPanelDisable().
		// For UART, we are using our own thread rather than the one within
		// tchmain, so when we get here, we can signal our thread to begin
		// monitoring for packets.
		if (NULL != gTouchDevice.hInitializationComplete)
		{
			SetEvent(gTouchDevice.hInitializationComplete);
		}
		else
		{
			CloseHandle(gTouchDevice.hUART);
			gTouchDevice.hUART = NULL;
		}
	}
}