/*****************************************************************************
* CODE OWNERSHIP AND DISCLAIMER OF LIABILITY
*
* Microchip Technology Incorporated ("Microchip") retains all ownership and
* intellectual property rights in the code accompanying this message and in
* all derivatives hereto.  You may use this code, and any derivatives created
* by any person or entity by or on your behalf, exclusively with Microchip's
* proprietary products.  Your acceptance and/or use of this code constitutes
* agreement to the terms and conditions of this notice.
*
* CODE ACCOMPANYING THIS MESSAGE IS SUPPLIED BY MICROCHIP "AS IS".  NO
* WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT
* LIMITED TO, IMPLIED WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND
* FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS CODE, ITS INTERACTION WITH
* MICROCHIP'S PRODUCTS, COMBINATION WITH ANY OTHER PRODUCTS, OR USE IN ANY
* APPLICATION.
*
* YOU ACKNOWLEDGE AND AGREE THAT, IN NO EVENT, SHALL MICROCHIP BE LIABLE,
* WHETHER IN CONTRACT, WARRANTY, TORT (INCLUDING NEGLIGENCE OR BREACH OF
* STATUTORY DUTY), STRICT LIABILITY, INDEMNITY, CONTRIBUTION, OR OTHERWISE,
* FOR ANY INDIRECT, SPECIAL, PUNITIVE, EXEMPLARY, INCIDENTAL OR CONSEQUENTIAL
* LOSS, DAMAGE, FOR COST OR EXPENSE OF ANY KIND WHATSOEVER RELATED TO THE
* CODE, HOWSOEVER CAUSED, EVEN IF MICROCHIP HAS BEEN ADVISED OF THE
* POSSIBILITY OR THE DAMAGES ARE FORESEEABLE.  TO THE FULLEST EXTENT ALLOWABLE
* BY LAW, MICROCHIP'S TOTAL LIABILITY ON ALL CLAIMS IN ANY WAY RELATED TO THIS
* CODE, SHALL NOT EXCEED THE PRICE YOU PAID DIRECTLY TO MICROCHIP SPECIFICALLY
* TO HAVE THIS CODE DEVELOPED.
*
* You agree that you are solely responsible for testing the code and
* determining its suitability.  Microchip has no obligation to modify, test,
* certify, or support the code.
*
* Author                Date        Comment
*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
* Steve Grahovac		03/30/10	Initial Microchip Touch CE example code
*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
* 
* Description:
*    This source file implements the platform specific code to enable communication
*    with the touch device.  The sections marked as platform specific may need
*    to the implemented to match the API of the target platform.  However, it is
*    recommended that the general logic and algorithms in the code that is not
*    platform specific be kept the same as much as possible.
* 
******************************************************************************/

#include <windows.h>
#include <types.h>
#include <nkintr.h>
#include <tchddsi.h>

// Platform specific headers
#include <ceddk.h>
#include <ceddkex.h>
#include <oal.h>
#include <oalex.h>
#include <omap3430.h>
#include <initguid.h>
#include <gpio.h>
#include <spi.h>
#include <i2c.h>
#include "Platform.h"
#include "touch.h"

static TOUCH_DEVICE TouchDevice =
{
	0, // GPIO Handle
	0, // SPI Handle
	0, // I2C Handle
	7  // GPIO ID to hook - Platform specific
};

// The following functions are platform specific and need to be modified to suit target platform's API

// Open GPIO bank
// GPIOOpen();

// Set input/ouput mode for GPIO pin
// GPIOSetMode();

// Retrieve GPIO hardware pin ID (different than GPIO ID)
// GPIOGetIrq();

// If necessary, additional configure pin so that it better detects low to high transition
// GPIOIoControl();

// Close handle to GPIO bank
// GPIOClose();

// Open handle to I2C port
// I2COpen();

// Set device address of current controller to communicate with
// I2CSetSlaveAddress();

// Set subaddress mode.  This offset is set to zero.
// I2CSetSubAddressMode();

// Set I2C speed to communicate at
// I2CSetBaudIndex();

// Write bytes to I2C device
// I2CWrite();

// Read bytes from I2C device
// I2CRead();

// Close handle to I2C port
// I2CClose();

// Open handle to SPI port
// SPIOpen();

// Configure SPI port to compatible communications protocol with device
// SPIConfigure();

// Read or write bytes to SPI device
// SPIWriteRead();

// Close handle to SPI port
// SPIClose();


/******************************************************************************
	Function:
		Init()

	Description:
		Initialize communications interface to interface with touch device.
******************************************************************************/
BOOL Init(UINT8 commInterface, LONG* receivedIRQIDToHook)
{
    DWORD  config;
	UINT8 numReceivedBytes=4;
	UINT32 registerStartAddress=0;
    IOCTL_GPIO_SET_DEBOUNCE_TIME_IN debounce;

	BYTE byte=0xFE;
	int index=0;

	DEBUGMSG(1, (TEXT("Init: begin %d\r\n"), commInterface));

	// open and initialize GPIO line so we may use it as an interrupt
    TouchDevice.hGPIO = GPIOOpen();
    if (TouchDevice.hGPIO == NULL)
    {
        DEBUGMSG(1, (TEXT("ERROR: Init: Failed open GPIO device driver\r\n")));
        return FALSE;
    }

	// Setup for input mode, falling edge detect, debounce enable
    GPIOSetMode(TouchDevice.hGPIO, TouchDevice.GPIOIDForTouchInterrupt, GPIO_DIR_INPUT|GPIO_INT_LOW_HIGH| GPIO_DEBOUNCE_ENABLE);
    // Get the IRQ ID from the GPIO ID
    *receivedIRQIDToHook = GPIOGetIrq(TouchDevice.hGPIO, TouchDevice.GPIOIDForTouchInterrupt);

	// Set debounce time on GPIO
    debounce.gpioId = TouchDevice.GPIOIDForTouchInterrupt;
    debounce.debounceTime = 10;
    GPIOIoControl(TouchDevice.hGPIO, IOCTL_GPIO_SET_DEBOUNCE_TIME, (UCHAR*)&debounce, sizeof(debounce), NULL, 0, NULL, NULL);

	if (COMMINTERFACE_I2C==commInterface)
	{
		TouchDevice.hI2C=NULL;
		TouchDevice.hI2C = I2COpen(I2CGetDeviceIdFromMembase(OMAP_I2C2_REGS_PA));

		if (TouchDevice.hI2C == NULL)
		  {
			DEBUGMSG(1, (TEXT("Failed open I2C device\r\n")));
			Destroy(commInterface);
			return FALSE;
		   }
		else
		{
			DEBUGMSG(1, (TEXT("Opened I2C device: 0x%x\r\n"),TouchDevice.hI2C));
		}

	    //initialization
		I2CSetSlaveAddress(TouchDevice.hI2C,  0x4d);
		I2CSetSubAddressMode(TouchDevice.hI2C, 0);
		// set at 100 khz
		I2CSetBaudIndex(TouchDevice.hI2C, SLOWSPEED_MODE);
	}
	else if (COMMINTERFACE_SPI==commInterface)
	{
		LONG    nSPIAddr=0;

		// Open SPI driver
		TouchDevice.hSPI = SPIOpen();
		if (TouchDevice.hSPI == NULL)
		{
			DEBUGMSG(1, (TEXT("ERROR: Init: Failed open SPI device driver\r\n")));
			Destroy(commInterface);
			return FALSE;
		}

		config =    MCSPI_PHA_EVEN_EDGES |
					MCSPI_POL_ACTIVEHIGH |
					MCSPI_CHCONF_CLKD(11) |
					MCSPI_CSPOLARITY_ACTIVELOW |
					MCSPI_CHCONF_WL(8) |
					MCSPI_CHCONF_TRM_TXRX |
					MCSPI_CHCONF_DMAW_DISABLE |
					MCSPI_CHCONF_DMAR_DISABLE |
					MCSPI_CHCONF_DPE0;

		// Configure SPI channel
		if (!SPIConfigure(TouchDevice.hSPI, nSPIAddr, config))
		{
			DEBUGMSG(1, (TEXT("ERROR: Init: Failed configure SPI device driver\r\n")));
			Destroy(commInterface);
			return FALSE;
		}
	}

	// Set controller to raw mode to maximize calibration accuracy
	if (COMMINTERFACE_SPI==commInterface)
	{
		UINT8 sendTouchOptionBytes[]={0x55,0x5,0x21,0x00,0x0d,0x01,0x00};
		UINT8 sendRegisterStartAddressBytes[]={0x55,0x1,0x22};
		UINT8 sendDisableBytes[]={0x55,0x1,0x13};
		UINT8 sendEnableBytes[]={0x55,0x1,0x12};
		UINT8 receiveBytes[10];
		UINT32 dummyValue=0;

		// clock out eight bytes without waiting for interrupt to flush SPI buffer
		numReceivedBytes=8;
		SPIWriteRead(TouchDevice.hSPI, 8, &dummyValue, receiveBytes);

		// Disable controller touch
		numReceivedBytes=4;
		SendCommandSPI(sendDisableBytes,3);
		Sleep(50);
		ReceiveResponseSPI(receiveBytes,&numReceivedBytes,250);

		numReceivedBytes=5;
		SendCommandSPI(sendRegisterStartAddressBytes,3);
		ReceiveResponseSPI(receiveBytes,&numReceivedBytes,250);
		registerStartAddress=receiveBytes[4];
		numReceivedBytes=4;
		// add register start address to modify register at appropriate offset
		sendTouchOptionBytes[4]+=registerStartAddress;
		SendCommandSPI(sendTouchOptionBytes,7);
		ReceiveResponseSPI(receiveBytes,&numReceivedBytes,250);

		// Enable controller touch
		numReceivedBytes=4;
		SendCommandSPI(sendEnableBytes,3);
		ReceiveResponseSPI(receiveBytes,&numReceivedBytes,250);
	}

	return TRUE;
}

/******************************************************************************
	Function:
		GetPacket()

	Description:
		Retrieve one packet from touch device.
******************************************************************************/
BOOL GetPacket(
	UINT8 commInterface,
    INT *xpos,
    INT *ypos,
	TOUCH_PANEL_SAMPLE_FLAGS *pTipStateFlags
)
{
	UINT32 control=0;
	UINT8 readBytes[5];
	int i;
	UINT8 numBytesToReceive=5;
	UINT8 numRetries=5;
	BOOL returnValue=FALSE;

	// initialize start byte
	for (i=0;i<5;i++)
	{
		readBytes[i]=0;
	}

	if (COMMINTERFACE_I2C==commInterface)
	{
		returnValue=ReceiveResponseI2C(readBytes, &numBytesToReceive, 250);
	}
	else if (COMMINTERFACE_SPI==commInterface)
	{
		returnValue=ReceiveResponseSPI(readBytes, &numBytesToReceive, 250);
	}

	if (TRUE==returnValue)
	{

		// packet has pen down bit set
		if (readBytes[0] & 1)
		{
			*pTipStateFlags=TouchSampleDownFlag | TouchSampleValidFlag;;
		}
		// packet does not have pen down bit set, so it is a pen up
		else
		{
			*pTipStateFlags=TouchSampleValidFlag;
		}

		*xpos= ((readBytes[2] & 0x1f)<<7) | (readBytes[1] & 0x7f);
		*ypos= ((readBytes[4] & 0x1f)<<7) | (readBytes[3] & 0x7f);

		for (i=0;i<5;i++)
		{
			DEBUGMSG(1,(TEXT("* 0x%x\r\n"),readBytes[i]));
		}

		DEBUGMSG(1,(TEXT("%d x %d y %d\r\n"),(1 & readBytes[0])!=0,*xpos,*ypos));
	}
	else
	{
		DEBUGMSG(1,(TEXT("GetPacket: Data read failure on packet read\r\n")));

		// If there was a communication error, set pen state to up to avoid a potential
		// stuck pen-down state
		*pTipStateFlags=TouchSampleValidFlag;
	}

	return TRUE;
}

/******************************************************************************
	Function:
		SendCommandI2C()

	Description:
		Send bytes to touch device using I2C interface.
******************************************************************************/
BOOL SendCommandI2C(UINT8* sendBytes,UINT8 numBytesToSend)
{
	UINT8 i;

	for (i=0;i<numBytesToSend;i++)
	{
		DEBUGMSG(1,(TEXT("SendCommand:Sending %x\r\n"),sendBytes[i]));
	}

	I2CSetSlaveAddress(TouchDevice.hI2C,  0x4d);

	// A small inter-byte delay in order for the touch device to correctly process data
	I2CWrite(TouchDevice.hI2C, I2C_SUBADDRESS_MODE_0, sendBytes,numBytesToSend);

	return TRUE;
}

/******************************************************************************
	Function:
		SendCommandSPI()

	Description:
		Send bytes to touch device using SPI interface.
******************************************************************************/
BOOL SendCommandSPI(UINT8* sendBytes,UINT8 numBytesToSend)
{
	UINT32 dummyValue=0;
	UINT8 i;
	BYTE udata[]={0x55,0x1,0x22};
	BOOL returnValue=FALSE;
	DWORD ret=0;

	DEBUGMSG(1,(TEXT("SendCommandSPI: begin\r\n")));

	for (i=0;i<numBytesToSend;i++)
	{
		DEBUGMSG(1,(TEXT("SendCommand:Sending %x\r\n"),sendBytes[i]));
	}


	// bytes are sent one at a time to introduce a small inter-byte delay
	for (i=0;i<numBytesToSend;i++)
	{
		returnValue=SPIWriteRead(TouchDevice.hSPI, 1, &sendBytes[i], &dummyValue);
		if (FALSE==returnValue)
		{
			break;
		}
	}

	return returnValue;
}

/******************************************************************************
	Function:
		ReceiveResponseI2C()

	Description:
		Receive bytes from touch device using I2C interface.
******************************************************************************/
BOOL ReceiveResponseI2C(UINT8* responseBytes, UINT8* numBytesToReceive, DWORD timeoutInMilliseconds)
{
	UINT32 dummyValue=0;
	UINT8 i;
	DWORD timeElapsed=0;

	DEBUGMSG(1,(TEXT("ReceiveResponseI2C: begin %u\r\n"),timeoutInMilliseconds));

	for (i=0;i<*numBytesToReceive;i++)
	{
		responseBytes[i]=0xff;
	}

	// wait for touch interrupt line to go high
	while ((0==GPIOGetBit(TouchDevice.hGPIO, TouchDevice.GPIOIDForTouchInterrupt)) && (timeElapsed<timeoutInMilliseconds))
	{
		Sleep(1);
		timeElapsed+=1;
	}


	if ((timeElapsed>=timeoutInMilliseconds) && (0!=timeoutInMilliseconds))
	{
		DEBUGMSG(1,(TEXT("ReceiveResponse: timed out\r\n")));
		return FALSE;
	}

	I2CSetSlaveAddress(TouchDevice.hI2C,  0x4d);
	I2CRead(TouchDevice.hI2C, I2C_SUBADDRESS_MODE_0, responseBytes, *numBytesToReceive);

	for (i=0;i<*numBytesToReceive;i++)
	{
		DEBUGMSG(1,(TEXT("%d-%x\r\n"),i,responseBytes[i]));
	}

	return TRUE;
}


/******************************************************************************
	Function:
		ReceiveResponseSPI()

	Description:
		Receive bytes from touch device using SPI interface.
******************************************************************************/
BOOL ReceiveResponseSPI(UINT8* responseBytes, UINT8* numBytesToReceive, DWORD timeoutInMilliseconds)
{
	UINT32 dummyValue=0;
	UINT8 i;
	DWORD timeElapsed=0;
	BOOL returnValue=FALSE;

	DEBUGMSG(1,(TEXT("ReceiveResponseSPI: begin\r\n")));
	for (i=0;i<*numBytesToReceive;i++)
	{
		responseBytes[i]=0;
	}

	// wait for touch interrupt line to go high
	while ((0==GPIOGetBit(TouchDevice.hGPIO, TouchDevice.GPIOIDForTouchInterrupt)) && (timeElapsed<timeoutInMilliseconds))
	{
		Sleep(1);
		timeElapsed+=1;
	}

	if (timeElapsed>=timeoutInMilliseconds)
	{
		DEBUGMSG(1,(TEXT("ReceiveResponse: timed out %d %d\r\n"),timeElapsed, timeoutInMilliseconds));
		return FALSE;
	}

	for (i=0;i<*numBytesToReceive;i++)
	{
		returnValue=SPIWriteRead(TouchDevice.hSPI, 1, &dummyValue, &responseBytes[i]);
		if (FALSE==returnValue)
		{
			DEBUGMSG(1,(TEXT("ReceiveResponse: Error receiving byte\r\n")));
			break;
		}
	}

	for (i=0;i<*numBytesToReceive;i++)
	{
		DEBUGMSG(1,(TEXT("%d-%x\r\n"),i,responseBytes[i]));
	}

	return TRUE;
}


/******************************************************************************
	Function:
		Destroy()

	Description:
		Close connection and free memory on current touch interface.
******************************************************************************/
void Destroy(UINT8 commInterface)
{
	if (COMMINTERFACE_SPI==commInterface)
	{
		if (TouchDevice.hSPI != NULL)
		{
			SPIClose(TouchDevice.hSPI);
			TouchDevice.hSPI = NULL;
		}
	}
	else if (COMMINTERFACE_I2C==commInterface)
	{
		if (TouchDevice.hI2C != NULL)
		{
			I2CClose(TouchDevice.hI2C);
			TouchDevice.hI2C = NULL;
		}
	}

    if (TouchDevice.hGPIO != NULL)
	{
		GPIOClose(TouchDevice.hGPIO);
		TouchDevice.hGPIO = NULL;
	}
}
